/*
Assignment #3 - You need to create an express HTTP server in Node.js which will handle the logic of a todo list app.

- Hard todo: Try to save responses in files, so that even if u exit the app and run it again, the data remains (similar to databases)
    Each todo has a title and a description. The title is a string and the description is a string.
    Each todo should also get an unique autogenerated id every time it is created
   
   The expected API endpoints are defined below,

1. GET `/todos` - Retrieve all todo items
   Description: Returns a list of all todo items.
   Response: 200 OK with an array of todo items in JSON format.
   Example: GET http://localhost:3000/todos

2. GET `/todos/:id` - Retrieve a specific todo item by ID
   Description: Returns a specific todo item identified by its ID.
   Response: 200 OK with the todo item in JSON format if found, or 404 Not Found if not found.
   Example: GET http://localhost:3000/todos/123

3. POST `/todos` - Create a new todo item
   Description: Creates a new todo item.
   Request Body: JSON object representing the todo item.
   Response: 201 Created with the ID of the created todo item in JSON format. eg: {id: 1}
   Example: POST http://localhost:3000/todos
   Request Body: { "title": "Buy groceries", "completed": false, description: "I should buy groceries" }

4. PUT `/todos/:id` - Update an existing todo item by ID
   Description: Updates an existing todo item identified by its ID.
   Request Body: JSON object representing the updated todo item.
   Response: 200 OK if the todo item was found and updated, or 404 Not Found if not found.
   Example: PUT http://localhost:3000/todos/123
   Request Body: { "title": "Buy groceries", "completed": true }

5. DELETE `/todos/:id` - Delete a todo item by ID
   Description: Deletes a todo item identified by its ID.
   Response: 200 OK if the todo item was found and deleted, or 404 Not Found if not found.
   Example: DELETE http://localhost:3000/todos/123
   
- For any other route not defined in the server return 404
*/

// import express module
const express = require("express");

// import body-parser module
const bodyParser = require("body-parser");

// import fs module
const fs = require("fs");

// create express app instance
const app = express();

// use body-parser middleware
app.use(bodyParser.json());

// define the port
const PORT = 3000;

// helper functions to find index of an object in an array and remove an object at a specific index
function findIndex(arr, id) {
    // find the index of the object with the given id in the array
    for (let i = 0; i < arr.length; i++) {
        
        // if the id matches, return the index
        if (arr[i].id === id) {
            return i;
        }
    }

    // if the id is not found, return -1
    return -1;
}

// helper function to remove an object at a specific index from an array
function removeAtIndex(arr, index) {
    // remove the object at the given index from the array
    let newArray = [];

    // iterate over the array and add all elements except the one at the given index
    for (let i = 0; i < arr.length; i++) {

        // if the index is not the one to be removed, add the element to the new array
        if (i !== index) {
            newArray.push(arr[i]);
        }
    }

    // return the new array with the object removed
    return newArray;
}

// define the routes to get all todos
app.get("/todos", (req, res) => {

    // read the todos from the file
    fs.readFile("todos.json", "utf8", function (err, data) {
        // if there is an error, throw it
        if (err) {
            throw err;
        }

        // send the todos as a response
        res.json(JSON.parse(data));
    });
});

// define the routes to get a specific todo by id
app.get("/todos/:id", (req, res) => {

    // read the todos from the file
    fs.readFile("todos.json", "utf8", function (err, data) {
        // if there is an error, throw it
        if (err) {
            throw err;
        }

        // parse the todos from the file
        const todos = JSON.parse(data);

        // find the index of the todo with the given id
        const todoIndex = findIndex(todos, parseInt(req.params.id));

        // if the todo is not found, return 404
        if (todoIndex === -1) {
            res.status(404).send();
        } else {
            // if the todo is found, return the todo
            res.json(todos[todoIndex]);
        }
    });
});

// define the routes to create a new todo
app.post("/todos", function (req, res) {

    // create a new todo object with the given title and description
    const newTodo = {
        id: Math.floor(Math.random() * 1000000), // unique random id
        title: req.body.title,
        description: req.body.description,
    };

    // read the todos from the file
    fs.readFile("todos.json", "utf8", (err, data) => {
        // if there is an error, throw it
        if (err) {
            throw err;
        }

        // parse the todos from the file
        const todos = JSON.parse(data);

        // add the new todo to the todos array
        todos.push(newTodo);

        // write the updated todos back to the file
        fs.writeFile("todos.json", JSON.stringify(todos), (err) => {
            // if there is an error, throw it
            if (err) {
                throw err;
            }

            // send the new todo as a response
            res.status(201).json(newTodo);
        });
    });
});

// define the routes to update a todo by id
app.put("/todos/:id", function (req, res) {

    // read the todos from the file
    fs.readFile("todos.json", "utf8", (err, data) => {
        // if there is an error, throw it
        if (err) {
            throw err;
        }

        // parse the todos from the file
        const todos = JSON.parse(data);

        // find the index of the todo with the given id
        const todoIndex = findIndex(todos, parseInt(req.params.id));

        // if the todo is not found, return 404
        if (todoIndex === -1) {
            res.status(404).send();
        } else {
            // update the title and description of the todo
            const updatedTodo = {
                id: todos[todoIndex].id,
                title: req.body.title,
                description: req.body.description,
            };

            // update the todo in the todos array
            todos[todoIndex] = updatedTodo;

            // write the updated todos back to the file
            fs.writeFile("todos.json", JSON.stringify(todos), (err) => {
                // if there is an error, throw it
                if (err) {
                    throw err;
                }

                // send the updated todo as a response
                res.status(200).json(updatedTodo);
            });
        }
    });
});

// define the routes to delete a todo by id
app.delete("/todos/:id", function (req, res) {

    // read the todos from the file
    fs.readFile("todos.json", "utf8", (err, data) => {
        // if there is an error, throw it
        if (err) {
            throw err;
        }

        // parse the todos from the file
        let todos = JSON.parse(data);

        // find the index of the todo with the given id
        const todoIndex = findIndex(todos, parseInt(req.params.id));

        // if the todo is not found, return 404
        if (todoIndex === -1) {
            res.status(404).send();
        } else {
            // remove the todo from the todos array
            todos = removeAtIndex(todos, todoIndex);

            // write the updated todos back to the file
            fs.writeFile("todos.json", JSON.stringify(todos), (err) => {
                // if there is an error, throw it
                if (err) {
                    throw err;
                }

                // send a success response
                res.status(200).send();
            });
        }
    });
});

// If the route is not defined, return 404
app.use((req, res, next) => {
    res.status(404).send();
});

// Start the server.
app.listen(PORT, () => {
    console.log(`Server is running on port ${PORT}`);
});
